// Main JavaScript file (/src/app/libry.js)
class SkillEnterprise {
    constructor() {
        this.API_ENDPOINT = 'https://techhub.com.ng/skent/src/ServerApi';
        this.currentUser = null;
        this.isAdmin = false;
        this.currentCourse = null;
        this.currentVideo = 0;
        this.autoplay = false;
        
        this.initializeEventListeners();
    }

    initializeEventListeners() {
        // Auth forms
        $('#loginForm').on('submit', (e) => this.handleLogin(e));
        $('#registerForm').on('submit', (e) => this.handleRegister(e));
        
        // Video controls
        $('#prevVideo').on('click', () => this.previousVideo());
        $('#nextVideo').on('click', () => this.nextVideo());
        $('#autoplay').on('change', (e) => this.toggleAutoplay(e));
        
        // Admin controls
        $('#addCourse').on('click', () => this.showAddCourseModal());
    }

    // Helper method for API calls
    async apiRequest(serverAction, method = 'POST', data = null) {
        const options = {
            url: this.API_ENDPOINT,
            method: method,
            data: { serveraction: serverAction }
        };

        if (data) {
            if (method === 'GET') {
                options.data = { ...options.data, ...data };
            } else {
                options.data = JSON.stringify({ ...data, serveraction: serverAction });
                options.contentType = 'application/json';
            }
        }

        try {
            const response = await $.ajax(options);
            if (!response.success) {
                throw new Error(response.message || 'Operation failed');
            }
            return response;
        } catch (error) {
            throw new Error(error.responseJSON?.message || error.message || 'Request failed');
        }
    }

    async handleLogin(e) {
        e.preventDefault();
        const formData = new FormData(e.target);
        
        try {
            const response = await this.apiRequest('login', 'POST', {
                email: formData.get('email'),
                password: formData.get('password')
            });
            
            if (response.success) {
                this.currentUser = response.data.user;
                this.isAdmin = response.data.user.role !== 'student';
                this.initializeDashboard();
            }
        } catch (error) {
            alert('Login failed: ' + error.message);
        }
    }

    async handleRegister(e) {
        e.preventDefault();
        const formData = new FormData(e.target);
        
        try {
            const response = await this.apiRequest('register', 'POST', {
                name: formData.get('name'),
                email: formData.get('email'),
                password: formData.get('password'),
                token: formData.get('token')
            });
            
            if (response.success) {
                alert('Registration successful! Please login.');
                $('#authModal').modal('hide');
            }
        } catch (error) {
            alert('Registration failed: ' + error.message);
        }
    }

    async loadCourse(courseId) {
        try {
            const response = await this.apiRequest('getCourse', 'GET', { courseId });
            
            this.currentCourse = response.data;
            this.currentVideo = response.data.lastProgress || 0;
            this.initializeVideoPlayer();
            this.updateProgressUI();
        } catch (error) {
            alert('Failed to load course: ' + error.message);
        }
    }

    async saveProgress() {
        try {
            await this.apiRequest('saveProgress', 'POST', {
                courseId: this.currentCourse.id,
                videoIndex: this.currentVideo,
                completed: true
            });
        } catch (error) {
            console.error('Failed to save progress:', error.message);
        }
    }

    // Admin functions
    async generateCourseToken(courseId, maxUsers) {
        try {
            const response = await this.apiRequest('generateToken', 'POST', {
                courseId,
                maxUsers
            });
            
            if (response.success) {
                alert(`Token generated: ${response.data.token}`);
            }
        } catch (error) {
            alert('Failed to generate token: ' + error.message);
        }
    }

    async addCourse(youtubePlaylistUrl) {
        try {
            const response = await this.apiRequest('addCourse', 'POST', {
                playlistUrl: youtubePlaylistUrl
            });
            
            if (response.success) {
                this.loadAdminDashboard();
            }
        } catch (error) {
            alert('Failed to add course: ' + error.message);
        }
    }

    // Video player methods
    initializeVideoPlayer() {
        const video = this.currentCourse.videos[this.currentVideo];
        $('#ytPlayer').attr('src', `https://www.youtube.com/embed/${video.youtube_id}`);
        this.updateVideoList();
    }

    updateVideoList() {
        const $list = $('#videoList');
        $list.empty();
        
        this.currentCourse.videos.forEach((video, index) => {
            const $item = $('<li>')
                .addClass('list-group-item')
                .text(video.title);
                
            if (index === this.currentVideo) {
                $item.addClass('active');
            }
            
            if (video.completed) {
                $item.append('<i class="fas fa-check text-success float-end"></i>');
            }
            
            $item.on('click', () => this.playVideo(index));
            
            $list.append($item);
        });
    }

    // Navigation methods
    previousVideo() {
        if (this.currentVideo > 0) {
            this.currentVideo--;
            this.initializeVideoPlayer();
        }
    }

    nextVideo() {
        if (this.currentVideo < this.currentCourse.videos.length - 1) {
            this.currentVideo++;
            this.initializeVideoPlayer();
        }
    }

    playVideo(index) {
        this.currentVideo = index;
        this.initializeVideoPlayer();
    }

    toggleAutoplay(e) {
        this.autoplay = e.target.checked;
    }
}

// Initialize the application
$(document).ready(() => {
    window.skillEnterprise = new SkillEnterprise();
});